

# login 
from fastapi import APIRouter, Depends, HTTPException
from sqlalchemy.orm import Session

from app.config.call_session import CallSession, session_maker
from app.controller.schema.user_schema import UserLoginModel, UserResponseModel
from app.database.repos.user_repo import UserRepo
from app.database.sql.models import User
import requests

from app.services.utils.user_access import check_if_user_is_super_user_or_project_owner

def login_user(
        user_login: UserLoginModel,
        session: CallSession = Depends(session_maker(auth=False)),
) -> UserResponseModel:
    """
    Login user and return user details.
    """
    # login in crm
    if not user_login.email or not user_login.password:
        raise HTTPException(
            status_code=400,
            detail="Email and password are required for login."
        )
    
    # check if user exists in CRM
    user_crm = login_to_crm(
        email=user_login.email,
        password=user_login.password,
    )

    if not user_crm:
        raise HTTPException(
            status_code=401,
            detail="Invalid email or password."
        )
    
    # check if user exists in database
    user = session[UserRepo].get_user_by_crm_id_and_email(
        user_id_crm=user_crm.get('id'),
        email=user_login.email,
    )

    if not user:
        # if user does not exist, create a new user
        user = User(
            email=user_login.email,
            user_id_crm=user_crm.get('id'),
        )
        session[UserRepo].add(user)
        session[UserRepo].commit()

    return UserResponseModel(
        id=user.id,
        email=user.email,
        user_id_crm=user.user_id_crm,
        cover=user_crm.get('cover', None),
        name=user_crm.get('name', None),
        bearer=user_crm.get('bearer', None),
    )

def list_users(
    project_id: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> list[UserResponseModel]:
    """
    List all users.
    """
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner(
        user_id_crm, project_id, session
    )
    users = session[UserRepo].get_all_users()
    return [UserResponseModel(
        id=user.id,
        email=user.email,
        user_id_crm=user.user_id_crm,
    ) for user in users]

def login_to_crm(
    email: str,
    password: str,
    crm_url: str = "https://www.crm.cri.org.ro/apicrm/intern",
) -> dict | None:
    """
    Login to CRM and return user details.
    """
    url = f"{crm_url}/login"
    form_data = {
        'user': email,
        'pass': password
    }

    try:
        response = requests.post(url, data=form_data)

        if response.ok:
            data = response.json()
            user = data[0]
            print("Login successful.")
            return user
        else:
            print("Invalid email or password.")
            return None
    except Exception as e:
        print(f"Login failed: {e}")
        return None
    
    