from datetime import datetime, timezone
import re
import numpy as np
from PIL import Image
from PIL import ImageDraw
from fastapi import Depends, File, HTTPException, UploadFile
from fastapi.responses import FileResponse
from sentinelhub import BBox, CRS
from sentinelhub import (
    SHConfig,
    SentinelHubRequest,
    DataCollection,
    MimeType,
    bbox_to_dimensions,
)

from app.config.call_session import CallSession, session_maker
from app.controller.schema.action_shema import (
    ActionAddModel,
    ActionResponseModel,
    ActionTypeAddModel,
    ActionSpeciesResponseModel,
)
from app.controller.schema.audit_action_shema import (
    AffectedZoneAddModel,
    AuditActionAddModel,
    AuditActionResponseModel,
)
from app.controller.schema.general_shema import MessageResponseModel, Stats
from app.controller.schema.project_shema import (
    HumidityReturnModel,
    ProjectAddModel,
    ProjectResponseModel,
)
from app.controller.schema.species_schema import SpecieInfo, SpeciesInfo
from app.controller.schema.tree_hole_detection import (
    TreeHoleDetectionAdd,
    TreeHoleDetectionReturn,
    TreeResponse,
)
from app.controller.schema.zone_shema import ProjectZoneAddModel, ProjectZoneUpdateModel
from app.database.repos.action_repo import ActionRepo
from app.database.repos.audit_action_repo import AuditActionRepo
from app.database.repos.file_repo import FileRepo
from app.database.repos.project_repo import ProjectRepo
from app.database.repos.species_repo import SpeciesRepo
from app.database.repos.tree_hole_detection_repo import TreeHoleDetectionRepo
from app.database.repos.tree_repo import TreeRepo
from app.database.repos.user_repo import UserRepo
from app.database.repos.zone_repo import ZoneRepo
from app.database.sql.models import RoleEnum, TreeDb, User
from app.services.utils.email_service import send_email
from app.services.utils.user_access import check_if_user_is_super_user, check_if_user_is_super_user_or_project_owner, check_if_user_is_super_user_or_project_owner_or_project_partner
from app.services.utils.utils import calculate_ha, get_lat_long, haversine, latlng_to_pixel, ramp
from geopy.distance import geodesic

def create_project(
    project_model: ProjectAddModel,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user(user_id_crm, session)

    project_db = session[ProjectRepo].add(project_model)
    if project_db:
        name = ""
        if project_model.city_name:
            name += f"{project_model.city_name}, "
        if project_model.county_name:
            name += f"{project_model.county_name}, "
        name += project_model.country_name + " " + str(project_db.project_id)
        session[ProjectRepo].update_project_name(project_db.project_id, name)

    for coordonate in project_model.area_coordonates:
        session[ProjectRepo].add_area(project_db.project_id, coordonate)

    if project_model.pedo_report:
        session[FileRepo].add(project_model.pedo_report, project_db.project_id)
    if project_model.area_coords:
        session[FileRepo].add(project_model.area_coords, project_db.project_id)
    for legal_doc in project_model.legal_docs:
        session[FileRepo].add(legal_doc, project_db.project_id)

    return MessageResponseModel(message=f"Project {project_db.project_id} created")


def get_project(
    project_id: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> ProjectResponseModel:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner_or_project_partner(
        user_id_crm, project_id, session
    )

    project_db = session[ProjectRepo].get_by_id(project_id)

    if project_db:
        project_dict = project_db.__dict__
        project_dict["created_at"] = project_db.created_at.isoformat()
        project_dict["updated_at"] = project_db.updated_at.isoformat()
        project_files = session[FileRepo].get_by_project_id(project_db.project_id)
        pedo_reports = [
            file.file for file in project_files if file.tag == "pedo-report"
        ]
        project_dict["pedo_report"] = pedo_reports[0] if pedo_reports else None
        project_dict["area_coords"] = [
            file.file for file in project_files if file.tag == "area-coords"
        ][0]
        project_dict["legal_docs"] = [
            file
            for file in project_files
            if file.tag != "pedo-report"
            and file.tag != "area-coords"
            and file.tag != "action-image"
        ]

        project_zones = session[ZoneRepo].get_by_project_id(project_db.project_id)
        project_dict["zones"] = []
        for zone in project_zones:
            zone_dict = zone.__dict__
            # get all species
            species = session[ZoneRepo].get_species_by_zone_id(zone.zone_id)

            species_return = []
            for specie in species:
                species_return.append(specie.__dict__)
            zone_dict["species"] = species_return
            zone_dict["number_of_actions"] = len(
                session[ActionRepo].get_action_zones_by_zone_id(zone.zone_id)
            )
            project_dict["zones"].append(zone_dict)

        project_dict["actions"] = get_actions(project_id, session)

        project_dict["audit_actions"] = get_audit_actions(project_id, session)

        project_dict["area_coordonates"] = []

        area_coords = session[ProjectRepo].get_areas(project_id)

        project_dict["area_coordonates"] = []
        for coord in area_coords:
            project_dict["area_coordonates"].append(coord.coordonates)

        return ProjectResponseModel.model_validate(project_dict)

    raise HTTPException(
        status_code=404,
        detail=f"Project {project_id} not found",
    )

def update_project(
    project_model: ProjectAddModel,
    project_id: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
    print(f"[DEBUG] Updating project {project_id}...")

    # Check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner(user_id_crm, project_id, session)

    # Fetch project
    project_db = session[ProjectRepo].get_by_id(project_id)
    if not project_db:
        raise HTTPException(status_code=404, detail="Project not found")
    print(f"[DEBUG] Found project: {project_db.project_name}")

    # Update main fields
    session[ProjectRepo].update_project(project_id, project_model)
    print("[DEBUG] Project base fields updated.")

    # Update project name
    name = ""
    if project_model.city_name:
        name += f"{project_model.city_name}, "
    if project_model.county_name:
        name += f"{project_model.county_name}, "
    name += project_model.country_name + f" {project_db.project_id}"
    session[ProjectRepo].update_project_name(project_db.project_id, name)
    print(f"[DEBUG] Updated project name: {name}")

    # ✅ Replace coordinates (hard delete)
    if project_model.area_coordonates is not None:
        print(f"[DEBUG] Replacing coordinates for project {project_id}...")

        # 1. Delete old coordinates completely
        old_areas = session[ProjectRepo].get_areas(project_id)
        deleted_count = 0
        for area in old_areas:
            session[ProjectRepo]._sql_session.delete(area)
            deleted_count += 1
        session[ProjectRepo]._sql_session.commit()
        print(f"[DEBUG] Deleted {deleted_count} old coordinates.")

        # 2. Add new ones (only valid coords)
        valid_coords = [
            coord for coord in project_model.area_coordonates
            if coord and "NaN" not in coord
        ]
        print(f"[DEBUG] Adding {len(valid_coords)} new valid coordinates...")
        for i, coord in enumerate(valid_coords):
            session[ProjectRepo].add_area(project_id, coord)
            print(f"[DEBUG] Added coord #{i + 1}: {coord[:100]}...")
        print("[DEBUG] Coordinates replaced successfully.")

    else:
        print("[DEBUG] No area coordinates provided — keeping existing ones.")

    # ✅ Replace files
    if project_model.area_coords:
        print("[DEBUG] Replacing area_coords file...")
        existing_files = session[FileRepo].get_by_project_id(project_id)
        for f in existing_files:
            if f.tag == "area-coords":
                session[FileRepo]._sql_session.delete(f)
        session[FileRepo]._sql_session.commit()
        session[FileRepo].add(project_model.area_coords, project_id)
        print("[DEBUG] New area_coords file added.")

    if project_model.pedo_report:
        print("[DEBUG] Updating pedo_report...")
        existing_files = session[FileRepo].get_by_project_id(project_id)
        for f in existing_files:
            if f.tag == "pedo-report":
                session[FileRepo]._sql_session.delete(f)
        session[FileRepo]._sql_session.commit()
        session[FileRepo].add(project_model.pedo_report, project_id)

    if project_model.legal_docs:
        print("[DEBUG] Replacing legal_docs...")
        existing_files = session[FileRepo].get_by_project_id(project_id)
        for f in existing_files:
            if f.tag not in ["pedo-report", "area-coords"]:
                session[FileRepo]._sql_session.delete(f)
        session[FileRepo]._sql_session.commit()
        for doc in project_model.legal_docs:
            session[FileRepo].add(doc, project_id)
        print(f"[DEBUG] Added {len(project_model.legal_docs)} new legal docs.")

    print(f"[SUCCESS] Project {project_id} updated successfully.")
    return MessageResponseModel(message=f"Project {project_id} updated successfully.")
def get_all_projects(
    zone_id: int = None,
    session: CallSession = Depends(session_maker(auth=True)),
) -> list[ProjectResponseModel]:
    projects = session[ProjectRepo].get_all()
    projects_return = []

    # filters
    zone = None
    if zone_id:
        zone = session[ZoneRepo].get_by_id(zone_id)

    for project in projects:
        # check if user has access to the project
        user_id_crm = session.user_id
        try:
            check_if_user_is_super_user_or_project_owner_or_project_partner(
                user_id_crm, project.project_id, session
            )
        except HTTPException:
            # if user is not super user or project owner or project partner, skip the project
            continue

        if zone_id and zone and zone.project_id != project.project_id:
            continue

        project_dict = project.__dict__
        project_dict["created_at"] = project.created_at.isoformat()
        project_dict["updated_at"] = project.updated_at.isoformat()
        project_files = session[FileRepo].get_by_project_id(project.project_id)
        pedo_reports = [
            file.file for file in project_files if file.tag == "pedo-report"
        ]
        project_dict["pedo_report"] = pedo_reports[0] if pedo_reports else None
        project_dict["area_coords"] = [
            file.file for file in project_files if file.tag == "area-coords"
        ][0]
        project_dict["legal_docs"] = [
            file
            for file in project_files
            if file.tag != "pedo-report"
            and file.tag != "area-coords"
            and file.tag != "action-image"
        ]

        project_zones = session[ZoneRepo].get_by_project_id(project.project_id)
        project_dict["zones"] = []
        for zone in project_zones:
            zone_dict = zone.__dict__
            # get all species
            species = session[ZoneRepo].get_species_by_zone_id(zone.zone_id)

            species_return = []
            for specie in species:
                species_return.append(specie.__dict__)
            zone_dict["species"] = species_return
            zone_dict["number_of_actions"] = len(
                session[ActionRepo].get_action_zones_by_zone_id(zone.zone_id)
            )
            project_dict["zones"].append(zone_dict)

        project_dict["actions"] = get_actions(project.project_id, session)

        area_coords = session[ProjectRepo].get_areas(project.project_id)
        project_dict["area_coordonates"] = []
        for coord in area_coords:
            project_dict["area_coordonates"].append(coord.coordonates)

        # project_dict["audit_actions"] = get_audit_actions(project.project_id, session)

        # add humidity ---------------------------------------------------------------------
        humidities = session[ProjectRepo].get_moisture(project.project_id)
        humidities_return = []

        for humidity in humidities:
            humidity_dict = humidity.__dict__
            humidity_dict["date"] = humidity.date.isoformat()
            humidities_return.append(HumidityReturnModel.model_validate(humidity_dict))

        # sort by date
        humidities_return.sort(
            key=lambda x: x.date, reverse=True if humidities_return else False
        )

        project_dict["humidities"] = humidities_return

        projects_return.append(ProjectResponseModel.model_validate(project_dict))
      
    return projects_return


def delete_project(
    project_id: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user(user_id_crm, session)
    
    session[ProjectRepo].delete(project_id)
    return MessageResponseModel(message=f"Project {project_id} deleted")


def create_zone(
    project_id: int,
    zone_model: ProjectZoneAddModel,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner(
        user_id_crm, project_id, session
    )


    # get all zones for a project
    project_zones = session[ZoneRepo].get_by_project_id(project_id)
    max_project_zone_id = 0
    for zone in project_zones:
        if int(zone.project_zone_id) > int(max_project_zone_id):
            max_project_zone_id = zone.project_zone_id
    max_project_zone_id = int(max_project_zone_id) + 1
    zone_db = session[ZoneRepo].add(project_id, zone_model, max_project_zone_id)
    if zone_db:
        for species in zone_model.species:
            session[ZoneRepo].add_species(zone_db.zone_id, species)
        return MessageResponseModel(message=f"Zone {zone_db.zone_id} created")
    else:
        return MessageResponseModel(message=f"Project {project_id} not found")


def update_zone(
    zone_id: int,
    zone_model: ProjectZoneUpdateModel,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
    # check permissions
    user_id_crm = session.user_id
    zone_db = session[ZoneRepo].get_by_id(zone_id)
    if not zone_db:
        raise HTTPException(
            status_code=404,
            detail=f"Zone {zone_id} not found",
        )
    project_id = zone_db.project_id
    check_if_user_is_super_user_or_project_owner(
        user_id_crm, project_id, session
    )


    if zone_db:
        zone_db_updated = session[ZoneRepo].update(zone_id, zone_model)
        if zone_db_updated:
            session[ZoneRepo].delete_species(zone_id)
            for species in zone_model.species:
                session[ZoneRepo].add_species(zone_db.zone_id, species)
        return MessageResponseModel(message=f"Zone {zone_id} updated")
    else:
        return MessageResponseModel(message=f"Zone {zone_id} not found")


def delete_zone(
    zone_id: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
        # check permissions
    user_id_crm = session.user_id
    zone_db = session[ZoneRepo].get_by_id(zone_id)
    if not zone_db:
        raise HTTPException(
            status_code=404,
            detail=f"Zone {zone_id} not found",
        )
    project_id = zone_db.project_id
    check_if_user_is_super_user_or_project_owner(
        user_id_crm, project_id, session
    )
    session[ZoneRepo].delete(zone_id)
    return MessageResponseModel(message=f"Zone {zone_id} deleted")


def add_action(
    project_id: int,
    action_model: ActionAddModel,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
    # check permissions 
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner(
        user_id_crm, project_id, session
    )

    action_db = session[ActionRepo].add(project_id, action_model)
    if action_db:
        # add images
        for image in action_model.images:
            session[FileRepo].add_action(image, action_db.action_id)
        # add action types
        for action_type in action_model.action_type:
            session[ActionRepo].add_action_type(action_db.action_id, action_type)
        # add zones
        for zone_id in action_model.zones:
            session[ActionRepo].add_action_zone(action_db.action_id, zone_id)
        return MessageResponseModel(message=f"Action {action_db.action_id} created")
    else:
        return MessageResponseModel(message=f"Project {project_id} not found")


def get_stats(
    session: CallSession = Depends(session_maker(auth=False)),
) -> Stats:
    area = 0.0
    saplings = 0
    zones = session[ZoneRepo].get_all()
    for zone in zones:
        species = session[ZoneRepo].get_species_by_zone_id(zone.zone_id)
        saplings += sum([specie.sapling_number for specie in species])

    projects = session[ProjectRepo].get_all()
    for project in projects:
        area_coords = session[ProjectRepo].get_areas(project.project_id)
        for coord in area_coords:
            coord = coord.coordonates.split(" ")
            coord = [tuple(map(float, coord.split(","))) for coord in coord]
            area += calculate_ha(coord)

    return Stats(
        number_of_projects=len(session[ProjectRepo].get_all()),
        number_of_saplings=saplings,
        number_of_actions=len(session[ActionRepo].get_all())
        + len(session[AuditActionRepo].get_all()),
        area=round(area, 2),
    )


def add_audit_action(
    project_id: int,
    audit_action_model: AuditActionAddModel,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner(
        user_id_crm, project_id, session
    )
    audit_action_db = session[AuditActionRepo].add(project_id, audit_action_model)
    if audit_action_db:
        # add biodiversity images
        for image in audit_action_model.biodiversity_images_urls:
            session[FileRepo].add_audit(image, audit_action_db.audit_action_id)
        # add images
        for image in audit_action_model.images:
            session[FileRepo].add_audit(image, audit_action_db.audit_action_id)
        # add action types
        for corrective_action_type in audit_action_model.corrective_actions:
            session[AuditActionRepo].add_action_type(
                audit_action_db.audit_action_id, corrective_action_type
            )
        # add affected zones
        for zone in audit_action_model.affected_zones:
            session[AuditActionRepo].add_affected_zone(
                audit_action_db.audit_action_id,
                zone.zone_id,
                zone.zone_name,
                zone.percentage,
            )

        # add volumes file
        if audit_action_model.volumes_file:
            session[FileRepo].add_audit(
                audit_action_model.volumes_file, audit_action_db.audit_action_id
            )

    else:
        return MessageResponseModel(message=f"Project {project_id} not found")
    return MessageResponseModel(message="Audit action created")


def get_audit_actions(
    project_id: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> list[AuditActionResponseModel]:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner_or_project_partner(
        user_id_crm, project_id, session
    )
    audit_actions = session[AuditActionRepo].get_by_project_id(project_id)
    audit_actions_return = []

    for audit_action in audit_actions:
        audit_action_dict = audit_action.__dict__
        audit_action_dict["created_at"] = audit_action.created_at.isoformat()
        audit_action_dict["updated_at"] = audit_action.updated_at.isoformat()
        audit_action_dict["deleted_at"] = audit_action.deleted_at.isoformat()
        audit_action_dict["date"] = audit_action.date.isoformat()
        audit_action_dict["action_id"] = audit_action.audit_action_id

        # get all biodiversity images
        biodiversity_images = session[FileRepo].get_by_audit_action_id(
            audit_action.audit_action_id,
            tag="biodiversity-image",
        )
        # audit_action_dict["biodiversity_images_urls"] is a FileAddModel
        audit_action_dict["biodiversity_images_urls"] = [
            image for image in biodiversity_images
        ]
        # get all images
        images = session[FileRepo].get_by_audit_action_id(
            audit_action.audit_action_id, tag="action-image"
        )
        audit_action_dict["images"] = [image for image in images]

        # get all action types
        action_types = session[AuditActionRepo].get_action_types_by_audit_action_id(
            audit_action.audit_action_id
        )
        audit_action_dict["corrective_actions"] = [
            ActionTypeAddModel(name=action_type.name, slug=action_type.slug)
            for action_type in action_types
        ]

        volumes_file = session[FileRepo].get_by_audit_action_id(
            audit_action.audit_action_id, tag="volumes-file"
        )
        if len(volumes_file) > 0:
            audit_action_dict["volumes_file"] = volumes_file[0]

        # proposed date
        if audit_action.propose_date:
            audit_action_dict["proposed_date"] = audit_action.propose_date.isoformat()

        # get all affected zones
        affected_zones = session[AuditActionRepo].get_affected_zones_by_audit_action_id(
            audit_action.audit_action_id
        )
        audit_action_dict["affected_zones"] = [
            AffectedZoneAddModel(
                percentage=zone.percentage,
                zone_id=zone.zone_id,
                zone_name=zone.zone_name,
            )
            for zone in affected_zones
        ]

        audit_actions_return.append(
            AuditActionResponseModel.model_validate(audit_action_dict)
        )

    return audit_actions_return


def get_actions(
    project_id: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> list[ActionResponseModel]:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner_or_project_partner(
        user_id_crm, project_id, session
    )
    actions = session[ActionRepo].get_by_project_id(project_id)
    actions_return = []

    for action in actions:
        action_dict = action.__dict__
        # get all images
        images = session[FileRepo].get_by_action_id(action.action_id)
        action_dict["images"] = [image for image in images]

        # get all action types
        action_types = session[ActionRepo].get_action_types_by_action_id(
            action.action_id
        )
        action_dict["action_type"] = [
            ActionTypeAddModel(name=action_type.name, slug=action_type.slug)
            for action_type in action_types
        ]

        # get all zones
        zones = session[ActionRepo].get_action_zones_by_action_id(action.action_id)
        action_dict["zones"] = [zone.zone_id for zone in zones]
        action_dict["species"] = [
            ActionSpeciesResponseModel(
                species_slug=specie.species_slug,
                species_name=specie.species_name,
                sapling_number=specie.sapling_number,
            )
            for specie in session[SpeciesRepo].get_species_for_action(action.action_id)
        ]
        action_dict["date"] = action.date.isoformat()
        action_dict["created_at"] = action.created_at.isoformat()
        action_dict["updated_at"] = action.updated_at.isoformat()
        action_dict["deleted_at"] = action.deleted_at.isoformat()

        actions_return.append(ActionResponseModel.model_validate(action_dict))

    return actions_return


def process_humidity(
    token: str,
    session: CallSession = Depends(session_maker(auth=False)),
) -> MessageResponseModel:
    if token != "12345":
        return MessageResponseModel(message="Invalid token")
    projects = session[ProjectRepo].get_all()
    for project in projects:
        # get all coordinates
        coordonates = session[ProjectRepo].get_areas(project.project_id)
        if not coordonates:
            continue

        # combine all coordinates
        area_coordonates = ""
        for coord in coordonates:
            area_coordonates += coord.coordonates + " "
        area_coordonates = area_coordonates.strip()

        coords = get_lat_long(area_coordonates)
        if len(coords) < 3:
            continue
        min_lon = min(coord[1] for coord in coords)
        max_lon = max(coord[1] for coord in coords)
        min_lat = min(coord[0] for coord in coords)
        max_lat = max(coord[0] for coord in coords)

        bounding_box = BBox(bbox=[min_lon, min_lat, max_lon, max_lat], crs=CRS.WGS84)

        config = SHConfig()
        config.sh_client_id = "51bdc9e1-897a-4676-85d2-a5cef862aacc"
        config.sh_client_secret = "0nJ0pr2AA80qa1oRvw7ZvxB6xDR93V71"

        resolution = 10  # Metri pe pixel
        size = bbox_to_dimensions(bounding_box, resolution=resolution)

        request = SentinelHubRequest(
            data_folder="data",
            evalscript="""
            //VERSION=3
            function setup() {
            return {
                input: ["B08", "B11"],
                output: { bands: 1, sampleType: "FLOAT32" } // Asigură-te că returnează un singur canal de tip FLOAT32
            };
            }

            function evaluatePixel(sample) {
            let ndmi = (sample.B08 - sample.B11) / (sample.B08 + sample.B11);
            return [ndmi];
            }

                """,
            input_data=[
                SentinelHubRequest.input_data(
                    data_collection=DataCollection.SENTINEL2_L2A,
                    mosaicking_order="mostRecent",
                )
            ],
            responses=[SentinelHubRequest.output_response("default", MimeType.TIFF)],
            bbox=bounding_box,
            size=size,
            config=config,
        )

        response = request.get_data()
        ndmi_image = response[0]

        new_image = np.zeros(
            (ndmi_image.shape[0], ndmi_image.shape[1], 3), dtype=np.uint8
        )
        for i in range(ndmi_image.shape[0]):
            for j in range(ndmi_image.shape[1]):
                new_image[i, j] = ramp(ndmi_image[i, j])

        # Convert the normalized array to a PIL Image in 'RGB' mode
        image = Image.fromarray(new_image, mode="RGB")

        image_width, image_height = image.size

        polygon_coords = get_lat_long(area_coordonates)

        # Convertim coordonatele poligonului în coordonate de pixeli
        polygon_pixels = [
            latlng_to_pixel(
                lat, lng, min_lat, max_lat, min_lon, max_lon, image_width, image_height
            )
            for lat, lng in polygon_coords
        ]
        mask = Image.new("L", (image_width, image_height), 0)
        ImageDraw.Draw(mask).polygon(polygon_pixels, outline=1, fill=255)

        # Creăm o imagine RGBA cu transparență completă (alpha = 0)
        output_image = Image.new("RGBA", (image_width, image_height), (0, 0, 0, 0))
        output_array = np.array(output_image)

        # Copiem conținutul original în output_image, doar unde masca este activă (255)
        original_array = np.array(image)
        mask_array = np.array(mask)

        # Setăm canalul alpha la 255 în zona de interes
        output_array[:, :, :3] = original_array[:, :, :3]
        output_array[:, :, 3] = mask_array

        # Convertim array-ul în imagine și salvăm
        final_image = Image.fromarray(output_array, "RGBA")
        final_image.save(f"data/{project.project_id}.png")

        # mean_moisture = ndmi_image.mean()
        ndmi_array = np.array(ndmi_image)
        masked_ndmi_values = ndmi_array[mask_array != 0]

        if masked_ndmi_values.size > 0:
            mean_moisture = masked_ndmi_values.mean()
        else:
            mean_moisture = 0

        min_value = -0.8
        max_value = 0.8

        mean_moisture_percentage = (
            (mean_moisture - min_value) / (max_value - min_value) * 100
        )
        mean_moisture_percentage = round(mean_moisture_percentage, 2)

        if mean_moisture_percentage < float(project.critical_moisture_level):
            # send critical email
            send_email(
                to_email="mihai.mihu@cri.org.ro",
                subject=f"Critical: Moisture for {project.project_name}",
                content=f"The project {project.project_name} hit the critical moisture warning level. Current moisture: {mean_moisture_percentage}%.\n\nPlease take a look on the map.",
            )

        elif mean_moisture_percentage < float(project.warning_moisture_level):
            # send warning email
            send_email(
                to_email="mihai.mihu@cri.org.ro",
                subject=f"Warning: Moisture for {project.project_name}",
                content=f"The project {project.project_name} hit the moisture warning level. Current moisture: {mean_moisture_percentage}%.\n\nPlease take a look on the map.",
            )

        session[ProjectRepo].add_moisture(project.project_id, mean_moisture_percentage)

    return MessageResponseModel(message="Humidity processed")


def get_humidity(
    project_id: int,
    session: CallSession = Depends(session_maker(auth=False)),
) -> list[HumidityReturnModel]:
    humidities = session[ProjectRepo].get_moisture(project_id)
    humidities_return = []

    for humidity in humidities:
        humidity_dict = humidity.__dict__
        humidity_dict["date"] = humidity.date.isoformat()
        humidities_return.append(HumidityReturnModel.model_validate(humidity_dict))

    # sort by date
    humidities_return.sort(
        key=lambda x: x.date, reverse=True if humidities_return else False
    )

    return humidities_return


def get_humidity_image(
    project_id: int,
    session: CallSession = Depends(session_maker(auth=False)),
) -> FileResponse:
    project = session[ProjectRepo].get_by_id(project_id)
    if not project:
        return None

    file_path = f"data/{project_id}.png"
    return FileResponse(
        file_path, media_type="image/jpeg", filename=f"{project_id}.png"
    )


def get_species_info(
    session: CallSession = Depends(session_maker(auth=False)),
) -> dict[str, SpecieInfo]:
    return session[SpeciesRepo].get_species_info()


def update_species_info(
    species: SpeciesInfo,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
        # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user(
        user_id_crm, session
    )
    for key, value in species.species_info.items():
        session[SpeciesRepo].update_specie_info(key, value)
    return MessageResponseModel(message="Species updated")


def tree_hole_detection(
    project_id: int,
    date: str,
    description: str,
    image: UploadFile = File(...),
    csv: UploadFile = File(...),
    tree_positions_csv: UploadFile = File(...),
    co2_file: UploadFile = File(...),
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner(
        user_id_crm, project_id, session
    )

    project = session[ProjectRepo].get_by_id(project_id)
    if not project:
        return MessageResponseModel(message=f"Project {project_id} not found")

    # save image to /data/holes_and_trees
    timestamp = datetime.now().strftime("%Y%m%d%H%M%S")
    image_filename = f"{project_id}_{timestamp}.png"
    with open(f"data/holes_and_trees/{image_filename}", "wb") as file:
        file.write(image.file.read())

    # save csv to /data/holes_and_trees
    csv_filename = f"{project_id}_{timestamp}.csv"
    with open(f"data/holes_and_trees/{csv_filename}", "wb") as file:
        file.write(csv.file.read())

    # open csv and count trees and holes
    trees = 0
    holes = 0
    with open(f"data/holes_and_trees/{csv_filename}", "r") as file:
        lines = file.readlines()
        for line in lines:
            if "tree" in line:
                trees += 1
            elif "hole" in line:
                holes += 1
    
    trees_co2_filename = f"{project_id}_{timestamp}_trees_co2.txt"
    co2_path = f"data/holes_and_trees/{trees_co2_filename}"
    with open(co2_path, "wb") as file:
        content_bytes = co2_file.file.read()
        file.write(content_bytes)

    # Decode as UTF-8 text to extract CO₂ value
    content = content_bytes.decode("utf-8", errors="ignore")

    match = re.search(r"CO₂ total:\s*([\d\.,]+)\s*kg/an", content)
    if match:
        co2_total = float(match.group(1).replace(",", "."))
        print(f"✅ CO₂ total extras: {co2_total} kg/an")
    else:
        co2_total = 0.0
        print("⚠️ Valoarea CO₂ total nu a fost găsită în fișier.")
    # insert detection record
    session[TreeHoleDetectionRepo].add(
        TreeHoleDetectionAdd(
            image_filename=image_filename,
            csv_filename=csv_filename,
            tree_number=trees,
            tree_hole_number=holes,
            description=description,
            date=date,
            created_at=datetime.now().isoformat(),
            updated_at=datetime.now().isoformat(),
            deleted_at=datetime.now().isoformat(),
            deleted=0,
            co2=co2_total,
            project_id=project_id,
        )
    )

    # save tree_positions_csv
    positions_csv_filename = f"{project_id}_{timestamp}_positions.csv"
    with open(f"data/holes_and_trees/{positions_csv_filename}", "wb") as file:
        file.write(tree_positions_csv.file.read())

    # read positions csv
    def haversine(lat1, lon1, lat2, lon2):
        import math
        R = 6371000  # meters
        phi1 = math.radians(lat1)
        phi2 = math.radians(lat2)
        delta_phi = math.radians(lat2 - lat1)
        delta_lambda = math.radians(lon2 - lon1)
        a = math.sin(delta_phi / 2) ** 2 + \
            math.cos(phi1) * math.cos(phi2) * math.sin(delta_lambda / 2) ** 2
        c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))
        return R * c

    existing_trees = session[TreeRepo].get_all_by_project(project_id)

    # Map existing trees by ID and by position
    existing_tree_ids = set(tree.tree_id for tree in existing_trees)
    processed_tree_ids = set()
    matched_existing_tree_ids = set()

    with open(f"data/holes_and_trees/{positions_csv_filename}", "r") as file:
        import csv
        reader = csv.DictReader(file)
        for row in reader:
            label = row.get("label", "").strip().lower()
            lon = float(row.get("lon", 0))
            lat = float(row.get("lat", 0))
            crown_area = float(row.get("crown_area", 0) or 0)
            height = float(row.get("height", 0) or 0)
            year_co2_quantity = float(row.get("year_co2_quantity", 0) or 0)

            tree_type = "sapling" if label == "hole" else "tree"

            found_close = False
            for existing_tree in existing_trees:
                if existing_tree.tree_id in matched_existing_tree_ids:
                    continue  # skip already matched trees

                try:
                    existing_lat = float(existing_tree.latitude)
                    existing_lon = float(existing_tree.longitude)
                except (ValueError, TypeError):
                    continue

                distance = haversine(lat, lon, existing_lat, existing_lon)
                print(
                    f"Matching CSV point ({lat:.8f}, {lon:.8f}) "
                    f"with DB tree ({existing_lat:.8f}, {existing_lon:.8f}) "
                    f"-> distance: {distance:.2f} m"
                )

                if distance <= 2.0:  # 2 meters threshold
                    # Update tree_type if different
                    if existing_tree.tree_type != tree_type:
                        existing_tree.tree_type = tree_type

                    # Update additional fields
                    if tree_type == "tree":
                        existing_tree.height = str(round(height, 2))
                        existing_tree.crown_area = round(crown_area, 3)
                        existing_tree.co2_year_kg = round(year_co2_quantity, 3)
                        diameter_m = 2 * (crown_area / np.pi) ** 0.5 if crown_area > 0 else None
                        existing_tree.diameter = str(round(diameter_m, 2)) if diameter_m else existing_tree.diameter

                    existing_tree.updated_at = datetime.now()
                    session[TreeRepo].add(existing_tree)

                    processed_tree_ids.add(existing_tree.tree_id)
                    matched_existing_tree_ids.add(existing_tree.tree_id)
                    found_close = True
                    break

            if not found_close:
                # Insert new tree or hole
                diameter_m = 2 * (crown_area / np.pi) ** 0.5 if crown_area > 0 else None
                new_tree = TreeDb(
                    species_name=None,
                    species_slug=None,
                    latitude=f"{lat:.8f}",
                    longitude=f"{lon:.8f}",
                    height=str(round(height, 2)) if tree_type == "tree" else "0.0",
                    diameter=str(round(diameter_m, 2)) if tree_type == "tree" else "0.0",
                    crown_area=round(crown_area, 3) if tree_type == "tree" else 0.0,
                    co2_year_kg=round(year_co2_quantity, 3) if tree_type == "tree" else 0.0,
                    age=None,
                    tree_type=tree_type,
                    project_id=project_id,
                    partner_id=None,
                    partner_name=None,
                    created_at=datetime.now(),
                    updated_at=datetime.now(),
                    deleted_at=datetime.now(),
                    deleted=0,
                )

                session[TreeRepo].add(new_tree)


    # Mark as dead trees not found in this CSV
    dead_tree_ids = existing_tree_ids - processed_tree_ids
    for tree_id in dead_tree_ids:
        tree = session[TreeRepo].get_by_id(tree_id)
        if tree and tree.tree_type != "dead":
            tree.tree_type = "dead"
            tree.updated_at = datetime.now()
            session[TreeRepo].add(tree)

    session[TreeRepo].commit()
    return MessageResponseModel(message="Tree hole detection added with tree positions processed")


def get_trees_by_project(
    project_id: int,    
    session: CallSession = Depends(session_maker(auth=False))
) -> list[TreeResponse]:
    trees = session[TreeRepo].get_all_by_project(project_id)
    trees_return = []
    for tree in trees:
        trees_return.append(TreeResponse.model_validate(tree.__dict__))
    
    return trees_return


def get_tree_hole_detection(
    project_id: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> list[TreeHoleDetectionReturn]:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner_or_project_partner(
        user_id_crm, project_id, session
    )
    tree_hole_detections = session[TreeHoleDetectionRepo].get_by_project_id(project_id)
    tree_hole_detections.sort(key=lambda x: x.date, reverse=True)
    tree_hole_detections_return = []
    for tree_hole_detection in tree_hole_detections:
        tree_hole_detection_dict = tree_hole_detection.__dict__
        tree_hole_detection_dict["created_at"] = (
            tree_hole_detection.created_at.isoformat()
        )
        tree_hole_detection_dict["updated_at"] = (
            tree_hole_detection.updated_at.isoformat()
        )
        tree_hole_detection_dict["deleted_at"] = (
            tree_hole_detection.deleted_at.isoformat()
        )
        tree_hole_detection_dict["id"] = tree_hole_detection.tree_hole_detection_id
        tree_hole_detection_dict["date"] = tree_hole_detection.date.isoformat()
        # add the server url to acces the image
        tree_hole_detection_dict["image_filename"] = (
            f"/data/holes_and_trees/{tree_hole_detection.image_filename}"
        )
        tree_hole_detection_dict["csv_filename"] = (
            f"/data/holes_and_trees/{tree_hole_detection.csv_filename}"
        )
        tree_hole_detection_dict["co2"] = tree_hole_detection.co2

        tree_hole_detections_return.append(
            TreeHoleDetectionReturn.model_validate(tree_hole_detection_dict)
        )

    return tree_hole_detections_return


def get_number_of_trees_available(
    project_id: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> int:
    # check permissions
    user_id_crm = session.user_id
    check_if_user_is_super_user_or_project_owner_or_project_partner(
        user_id_crm, project_id, session
    )
    trees = session[TreeRepo].get_all_not_assigned_by_project(project_id)
    return len(trees)


def assign_trees_to_partner(
    project_id: int,
    partner_id: int,
    number_of_trees: int,
    session: CallSession = Depends(session_maker(auth=True)),
) -> MessageResponseModel:
    user_id_crm = session.user_id
    print(f"Assigning {number_of_trees} trees to partner {partner_id} for project {project_id}")
    check_if_user_is_super_user_or_project_owner(
        user_id_crm, project_id, session
    )

    
    partner = session[UserRepo].get_by_id(partner_id)
    if not partner:
        return MessageResponseModel(message=f"Partner {partner_id} not found")

    all_trees = session[TreeRepo].get_all_not_assigned_by_project(project_id)
    if not all_trees:
        return MessageResponseModel(message="No trees to assign")

    if number_of_trees > len(all_trees):
        return MessageResponseModel(
            message=f"Not enough trees to assign. Available: {len(all_trees)}, requested: {number_of_trees}"
        )

    # Convertim lat/lon și ignorăm copacii cu poziții invalide
    valid_trees = []
    for tree in all_trees:
        try:
            lat = float(tree.latitude)
            lon = float(tree.longitude)
            valid_trees.append((tree, (lat, lon)))
        except (ValueError, TypeError):
            continue

    if len(valid_trees) < number_of_trees:
        return MessageResponseModel(
            message=f"Not enough trees with valid location. Available: {len(valid_trees)}, requested: {number_of_trees}"
        )

    # Sortăm toți copacii după densitate spațială (cluster de copaci cei mai apropiați)
    def find_best_cluster(trees_with_coords, k):
        best_group = []
        min_avg_distance = float("inf")

        for i in range(len(trees_with_coords)):
            center_tree, center_coord = trees_with_coords[i]
            # calculăm distanța către toți ceilalți
            distances = sorted(
                trees_with_coords,
                key=lambda x: geodesic(center_coord, x[1]).meters
            )
            cluster = distances[:k]
            avg_distance = sum(
                geodesic(center_coord, c[1]).meters for c in cluster
            ) / k

            if avg_distance < min_avg_distance:
                min_avg_distance = avg_distance
                best_group = [c[0] for c in cluster]

        return best_group

    selected_trees = find_best_cluster(valid_trees, number_of_trees)

    # Asignăm copacii partenerului
    for tree in selected_trees:
        tree.partner_id = partner.id
        tree.partner_name = partner.email
        tree.updated_at = datetime.now(timezone.utc)
        session[TreeRepo].add(tree)
    
    session[TreeRepo].commit()
    # if user is not project_partner, project_owner or super_user, make it project_partner
    user_project_role = session[UserRepo].get_user_project_role(
        user_id_crm, project_id
    )

    if user_project_role is None or user_project_role.role != RoleEnum.project_partner or user_project_role.role != RoleEnum.project_owner:
        session[UserRepo].add_user_project_role(
            partner_id, project_id, RoleEnum.project_partner
        )
        session[UserRepo].commit()

    return MessageResponseModel(message=f"{number_of_trees} trees successfully assigned to partner.")