from datetime import datetime, timedelta
from datetime import timezone

import jwt
from fastapi import HTTPException, Security
from fastapi.security import HTTPAuthorizationCredentials, HTTPBearer
from passlib.context import CryptContext

from app.config import auth_settings


class AuthHandler:
    """Handles authentication and authorization, hashing and token generation"""

    security = HTTPBearer()

    def __init__(self):
        self._pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")
        self._settings = auth_settings

    def get_hash(self, password: str) -> str:
        """Hashes using encryption algorithm"""
        return self._pwd_context.hash(password)

    def compare(self, plain_text: str, hashed_text: str) -> bool:
        """Compares plain text and hashed text"""
        return self._pwd_context.verify(plain_text, hashed_text)

    def decode_token(self, token: str) -> dict:
        """Decodes the token using the secret key and returns the payload"""
        try:
            return jwt.decode(token, self._settings.SECRET_KEY, algorithms=["HS256"])
        except jwt.ExpiredSignatureError as e:
            raise HTTPException(status_code=401, detail="Token expired") from e
        except jwt.InvalidTokenError as e:
            raise HTTPException(status_code=401, detail="Invalid token") from e

    def generate_auth_token(self, user_id, user_role) -> str:
        """Generates a token using the user id"""
        payload = {
            "exp": datetime.now(timezone.utc)
            + timedelta(days=self._settings.AUTH_TOKEN_EXPIRE_DAYS),
            "iat": datetime.now(timezone.utc),
            "sub": {"user_id": user_id, "user_role": user_role},
        }
        return jwt.encode(payload, self._settings.SECRET_KEY, algorithm="HS256")

    def auth_wrapper(self, auth: HTTPAuthorizationCredentials = Security(security)):
        payload = self.decode_token(auth.credentials)
        try:
            user_id = payload["data"]["user_id"]
            partner_id = payload["data"]["partner_id"]
        except ValueError as e:
            raise HTTPException(status_code=401, detail="Invalid token") from e
        return user_id, partner_id

    def generate_email_confirmation_token(self, email: str) -> str:
        """Generates a token using the email and expiration time"""
        payload = {
            "email": email,
            "exp": datetime.now(timezone.utc) + timedelta(days=30),
            "iat": datetime.now(timezone.utc),
        }
        return jwt.encode(payload, self._settings.SECRET_KEY, algorithm="HS256")

    def decode_email_confirmation_token(self, token) -> str:
        """Decodes the token using the secret key and returns the email"""
        try:
            return jwt.decode(token, self._settings.SECRET_KEY, algorithms=["HS256"])[
                "email"
            ]
        except jwt.ExpiredSignatureError as e:
            raise HTTPException(status_code=401, detail="Token expired") from e
        except jwt.InvalidTokenError as e:
            raise HTTPException(status_code=401, detail="Invalid token") from e

    def generate_reset_password_token(self, email: str) -> str:
        """Generates a token using the email and expiration time"""
        payload = {
            "email": email,
            "exp": datetime.now(timezone.utc)
            + timedelta(minutes=self._settings.RESET_PASSWORD_TOKEN_EXPIRE_MINUTES),
            "iat": datetime.now(timezone.utc),
        }
        return jwt.encode(payload, self._settings.SECRET_KEY, algorithm="HS256")

    def decode_reset_password_token(self, token) -> str:
        """Decodes the token using the secret key and returns the email"""
        try:
            return jwt.decode(token, self._settings.SECRET_KEY, algorithms=["HS256"])[
                "email"
            ]
        except jwt.ExpiredSignatureError as e:
            raise HTTPException(status_code=401, detail="Token expired") from e
        except jwt.InvalidTokenError as e:
            raise HTTPException(status_code=401, detail="Invalid token") from e
