import contextlib
from typing import Iterator

from sqlalchemy import Engine, create_engine
from sqlalchemy.orm import sessionmaker, Session

from .models import Base
from ...config import sql_database_settings
from ...security.auth_handler import AuthHandler

auth_handler = AuthHandler()


class DatabaseSessionManager:
    def __init__(self, base):
        self._engine: Engine | None = None
        self._session_maker: sessionmaker | None = None
        self.base = base

    def init(self, host: str = sql_database_settings.CONNECTION_URL):
        self._engine = create_engine(host, pool_pre_ping=True)
        self._session_maker = sessionmaker(autocommit=False, bind=self._engine)
        self.base.metadata.create_all(bind=self._engine)

        self._ensure_default()

    def _ensure_default(self):
        if self._engine is None:
            raise Exception("DatabaseSessionManager is not initialized")
        if self._session_maker is None:
            raise Exception("DatabaseSessionManager is not initialized")

        # ensure defaults

    def close(self):
        if self._engine is None:
            raise Exception("DatabaseSessionManager is not initialized")
        self._engine.dispose()
        self._engine = None
        self._session_maker = None

    @contextlib.contextmanager
    def session(self) -> Iterator[Session]:
        if self._session_maker is None:
            raise Exception("DatabaseSessionManager is not initialized")

        session = self._session_maker()
        try:
            yield session
        except Exception:
            session.rollback()
            raise
        finally:
            session.close()


sqlDBManager = DatabaseSessionManager(base=Base)
