from datetime import datetime
from app.controller.schema.zone_shema import ProjectZoneAddModel, SpeciesAddModel
from app.database.repos.base_repo import SqlRepo
from app.database.sql.models import ProjectZoneDb, SpeciesDb


class ZoneRepo(SqlRepo):
    def __init__(self, app_session):
        super().__init__(app_session)

    def add(
        self, project_id: int, zone_model: ProjectZoneAddModel, project_zone_id: int
    ) -> ProjectZoneDb:
        zone_db = ProjectZoneDb(
            partner_id=zone_model.partner_id,
            partner_name=zone_model.partner_name,
            co2=zone_model.co2,
            model=zone_model.model,
            field_schema=zone_model.field_schema,
            zone_coordonates=zone_model.zone_coordonates,
            project_id=project_id,
            deleted=0,
            created_at=datetime.now(),
            project_zone_id=project_zone_id,
        )
        self._sql_session.add(zone_db)
        self._sql_session.commit()
        self._sql_session.refresh(zone_db)
        return zone_db

    def update(self, zone_id: int, zone_model: ProjectZoneAddModel) -> ProjectZoneDb:
        zone_db = self.get_by_id(zone_id)
        if zone_db:
            zone_db.co2 = zone_model.co2
            zone_db.model = zone_model.model
            zone_db.field_schema = zone_model.field_schema
            zone_db.zone_coordonates = zone_model.zone_coordonates
            self._sql_session.commit()
        else:
            raise ValueError("Zone not found")
        return zone_db

    def delete(self, zone_id: int) -> None:
        zone_db = self.get_by_id(zone_id)
        zone_db.deleted = 1
        zone_db.deleted_at = datetime.now()
        self._sql_session.commit()

    def get_all(self) -> list[ProjectZoneDb]:
        return (
            self._sql_session.query(ProjectZoneDb)
            .filter(ProjectZoneDb.deleted == 0)
            .all()
        )

    def get_by_id(self, zone_id: int) -> ProjectZoneDb:
        return (
            self._sql_session.query(ProjectZoneDb)
            .filter(ProjectZoneDb.zone_id == zone_id)
            .first()
        )

    def add_species(self, zone_id: int, species: SpeciesAddModel) -> ProjectZoneDb:
        zone_db = self.get_by_id(zone_id)
        if zone_db:
            species_db = SpeciesDb(
                species_name=species.species_name,
                species_slug=species.species_slug,
                sapling_number=species.sapling_number,
                project_zone_id=zone_id,
            )
            self._sql_session.add(species_db)
            self._sql_session.commit()
            self._sql_session.refresh(species_db)
        else:
            raise ValueError("Zone not found")

        return species_db

    def delete_species(self, zone_id: int) -> None:
        self._sql_session.query(SpeciesDb).filter(
            SpeciesDb.project_zone_id == zone_id
        ).delete()
        self._sql_session.commit()

    def get_by_project_id(self, project_id: int) -> list[ProjectZoneDb]:
        return (
            self._sql_session.query(ProjectZoneDb)
            .filter(ProjectZoneDb.project_id == project_id)
            .filter(ProjectZoneDb.deleted == 0)
            .all()
        )

    def get_species_by_zone_id(self, zone_id: int) -> list[SpeciesDb]:
        return (
            self._sql_session.query(SpeciesDb)
            .filter(SpeciesDb.project_zone_id == zone_id)
            .all()
        )

    def delete_by_project_id(self, project_id: int):
        # for each zone, delete species
        zones = self.get_by_project_id(project_id)
        for zone in zones:
            self._sql_session.query(SpeciesDb).filter(
                SpeciesDb.project_zone_id == zone.zone_id
            ).delete()
        # delete zones
        self._sql_session.query(ProjectZoneDb).filter(
            ProjectZoneDb.project_id == project_id
        ).delete()
        self._sql_session.commit()
        return True

    def delete_by_zone_id(self, zone_id: int) -> None:
        # set deleted flag
        zone_db = self.get_by_id(zone_id)
        zone_db.deleted = 1
        zone_db.deleted_at = datetime.now()
        self._sql_session.commit()
