from app.database.repos.base_repo import SqlRepo
from app.database.sql.models import RoleEnum, TreeDb, User, UserProjectRole


class UserRepo(SqlRepo):

    def get_user_by_crm_id_and_email(self, user_id_crm: str, email: str):
        return self._sql_session.query(User).filter(
            User.user_id_crm == user_id_crm,
            User.email == email
        ).first()

    def get_all_users(self):
        return self._sql_session.query(User).all()

    def get_user_by_crm_id(self, user_id_crm: str):
        return self._sql_session.query(User).filter(
            User.user_id_crm == user_id_crm
        ).first()
    
    def get_by_id(self, user_id: int):
        return self._sql_session.query(User).get(user_id)

    def add(self, user: User):
        self._sql_session.add(user)

    def add_user_project_role(self, user_id: int, project_id: int, role: str):
        user_project_role = UserProjectRole(user_id=user_id, project_id=project_id, role=role)
        self._sql_session.add(user_project_role)
        return user_project_role

    def is_super_user(self, user_id_crm: str) -> bool:
        user = self._sql_session.query(User).filter(User.user_id_crm == user_id_crm).first()
        user_project_role = self._sql_session.query(UserProjectRole).filter(UserProjectRole.user_id == user.id).all()
        if not user_project_role:
            return False
        for role in user_project_role:
            if role.role == "super-user":
                return True
            
        return False
    
    def get_user_project_role(self, user_id_crm: str, project_id: int) -> UserProjectRole | None:
        user = self._sql_session.query(User).filter(User.user_id_crm == user_id_crm).first()
        if not user:
            return None
        
        user_project_role = self._sql_session.query(UserProjectRole).filter(
            UserProjectRole.user_id == user.id,
            UserProjectRole.project_id == project_id
        ).first()
        
        return user_project_role

    def get_all_users_with_roles(self):
        users = (
            self.db.query(User)
            .outerjoin(UserProjectRole, User.id == UserProjectRole.user_id)
            .all()
        )
        result = []
        for user in users:
            roles = [r.role for r in user.user_project_roles]
            result.append({
                "id": user.id,
                "name": user.name,
                "email": user.email,
                "roles": roles,
            })
        return result

    def set_super_user(self, user_id: int):
        user = self.db.query(User).filter(User.id == user_id).first()
        if not user:
            raise Exception("User not found.")
        user.is_super_user = True
        self.db.commit()

    def set_user_project_role(self, user_id: int, project_id: int, role: RoleEnum):
        existing = (
            self.db.query(UserProjectRole)
            .filter_by(user_id=user_id, project_id=project_id)
            .first()
        )
        if existing:
            existing.role = role
        else:
            new_role = UserProjectRole(
                user_id=user_id, project_id=project_id, role=role
            )
            self.db.add(new_role)
        self.db.commit()

    def commit(self):
        self._sql_session.commit()