from datetime import datetime
from app.controller.schema.tree_hole_detection import TreeHoleDetectionAdd
from app.database.repos.base_repo import SqlRepo
from app.database.sql.models import TreeHoleDetectionDb


class TreeHoleDetectionRepo(SqlRepo):
    def __init__(self, app_session):
        super().__init__(app_session)

    def get_all(self) -> list[TreeHoleDetectionDb]:
        return self._sql_session.query(TreeHoleDetectionDb).all()

    def add(self, tree_hole_detection: TreeHoleDetectionAdd):
        created_at = datetime.fromisoformat(tree_hole_detection.created_at)
        updated_at = datetime.fromisoformat(tree_hole_detection.updated_at)
        deleted_at = (
            datetime.fromisoformat(tree_hole_detection.deleted_at)
            if tree_hole_detection.deleted_at
            else None
        )
        tree_hole_detection_db = TreeHoleDetectionDb(
            image_filename=tree_hole_detection.image_filename,
            csv_filename=tree_hole_detection.csv_filename,
            tree_number=tree_hole_detection.tree_number,
            tree_hole_number=tree_hole_detection.tree_hole_number,
            description=tree_hole_detection.description,
            date=self.convert_date(tree_hole_detection.date),
            created_at=created_at,
            updated_at=updated_at,
            deleted_at=deleted_at,
            co2=tree_hole_detection.co2,
            deleted=tree_hole_detection.deleted,
            project_id=tree_hole_detection.project_id,
        )
        self._sql_session.add(tree_hole_detection_db)
        self._sql_session.commit()
        self._sql_session.refresh(tree_hole_detection_db)
        return tree_hole_detection_db

    def convert_date(self, date_str: str):
        return datetime.strptime(date_str, "%Y-%m-%d %H:%M:%S.%f")

    def get_by_project_id(self, project_id: int) -> list[TreeHoleDetectionDb]:
        return (
            self._sql_session.query(TreeHoleDetectionDb)
            .filter(TreeHoleDetectionDb.project_id == project_id)
            .all()
        )
