from datetime import datetime
from app.controller.schema.project_shema import ProjectAddModel
from app.database.repos.base_repo import SqlRepo
from app.database.sql.models import HumidityDb, ProjectCoordonatesDb, ProjectDb, ProjectPartnerDb


class ProjectRepo(SqlRepo):
    def __init__(self, app_session):
        super().__init__(app_session)

    def get_all(self) -> list[ProjectDb]:
        return self._sql_session.query(ProjectDb).filter(ProjectDb.deleted == 0).all()

    def get_by_id(self, project_id: int) -> ProjectDb:
        return (
            self._sql_session.query(ProjectDb)
            .filter(ProjectDb.project_id == project_id)
            .first()
        )

    def get_by_partner_id(self, partner_id: int) -> list[ProjectDb]:
        return (
            self._sql_session.query(ProjectDb)
            .filter(ProjectDb.partner_id == partner_id)
            .filter(ProjectDb.deleted == 0)
            .all()
        )
    
    def get_projects_ids_by_partner_id(self, partner_id: int) -> list[int]:
        project_patners = self._sql_session.query(ProjectPartnerDb).filter(
            ProjectPartnerDb.partner_id == partner_id,
            ProjectPartnerDb.deleted == 0
        ).all()
        return [project_partner.project_id for project_partner in project_patners]
    

    def add_area(self, project_id: int, area_coordonates: str) -> ProjectCoordonatesDb:
        project_coordonates_db = ProjectCoordonatesDb(
            project_id=project_id,
            coordonates=area_coordonates,
            created_at=datetime.now(),
            updated_at=datetime.now(),
            deleted=0,
            deleted_at=datetime.now(),
        )
        self._sql_session.add(project_coordonates_db)
        self._sql_session.commit()
        self._sql_session.refresh(project_coordonates_db)
        return project_coordonates_db

    def get_areas(self, project_id: int) -> list[ProjectCoordonatesDb]:
        return (
            self._sql_session.query(ProjectCoordonatesDb)
            .filter(ProjectCoordonatesDb.project_id == project_id)
            .filter(ProjectCoordonatesDb.deleted == 0)
            .all()
        )

    def add(self, project: ProjectAddModel) -> ProjectDb:
        project_db = ProjectDb(
            partner_id=project.partner_id,
            project_name="No name",
            partner_name=project.partner_name,
            country_id=project.country_id,
            country_name=project.country_name,
            county_id=project.county_id,
            county_name=project.county_name,
            city_id=project.city_id,
            city_name=project.city_name,
            soil_type=project.soil_type,
            type=project.type,
            co2=project.co2,
            social_goal=project.social_goal,
            warning_moisture_level=project.warning_moisture_level,
            critical_moisture_level=project.critical_moisture_level,
            created_at=datetime.now(),
            updated_at=datetime.now(),
            deleted=0,
            deleted_at=datetime.now(),
        )
        self._sql_session.add(project_db)
        self._sql_session.commit()
        self._sql_session.refresh(project_db)

        return project_db

    def update_project(
        self, project_id: int, project_model: ProjectAddModel
    ) -> ProjectDb:
        project_db = self.get_by_id(project_id)
        project_db.partner_id = project_model.partner_id
        project_db.partner_name = project_model.partner_name
        project_db.country_id = project_model.country_id
        project_db.country_name = project_model.country_name
        project_db.county_id = project_model.county_id
        project_db.county_name = project_model.county_name
        project_db.city_id = project_model.city_id
        project_db.city_name = project_model.city_name
        project_db.soil_type = project_model.soil_type
        project_db.warning_moisture_level = project_model.warning_moisture_level
        project_db.critical_moisture_level = project_model.critical_moisture_level
        project_db.type = project_model.type
        project_db.co2 = project_model.co2
        project_db.social_goal = project_model.social_goal
        project_db.area_coordonates = project_model.area_coordonates
        project_db.updated_at = datetime.now()
        self._sql_session.commit()
        self._sql_session.refresh(project_db)
        return project_db

    def update_project_name(self, project_id: int, project_name: str) -> ProjectDb:
        project_db = self.get_by_id(project_id)
        project_db.project_name = project_name
        project_db.updated_at = datetime.now()
        self._sql_session.commit()
        self._sql_session.refresh(project_db)
        return project_db

    def delete(self, project_id: int) -> ProjectDb:
        project_db = self.get_by_id(project_id)
        project_db.deleted = 1
        project_db.deleted_at = datetime.now()
        self._sql_session.commit()
        return project_db

    def get_project_area(self, project_id: int) -> str:
        project_db = self.get_by_id(project_id)
        return project_db.area_coordonates

    def add_moisture(self, project_id: int, moisture: str) -> HumidityDb:
        humidity_db = HumidityDb(
            project_id=project_id, humidity=moisture, date=datetime.now()
        )
        self._sql_session.add(humidity_db)
        self._sql_session.commit()
        self._sql_session.refresh(humidity_db)
        return humidity_db

    def get_moisture(self, project_id: int) -> list[HumidityDb]:
        return (
            self._sql_session.query(HumidityDb)
            .filter(HumidityDb.project_id == project_id)
            .all()
        )
        
    def commit(self):
        self._sql_session.commit()
