from datetime import datetime

from app.controller.schema.action_shema import ActionAddModel, ActionTypeAddModel
from app.database.repos.base_repo import SqlRepo
from app.database.sql.models import ActionDb, ActionTypeDb, ActionZoneMTM, SpeciesDb


class ActionRepo(SqlRepo):
    def __init__(self, app_session):
        super().__init__(app_session)

    def add(self, project_id: int, action_model: ActionAddModel) -> ActionDb:
        plant_tree_action = any(
            [action for action in action_model.action_type if action.slug == "planting-tree-action"]
        )

        if plant_tree_action and len(action_model.action_type) > 1:
            raise ValueError("Planting tree action can't have other actions")

        action_db = ActionDb(
            project_id=project_id,
            date=action_model.date,
            sapling_number=action_model.sapling_number,
            video_link=action_model.video_link,
            deleted=0,
            deleted_at=datetime.now(),
            created_at=datetime.now(),
            updated_at=datetime.now(),
        )
        self._sql_session.add(action_db)
        self._sql_session.commit()
        self._sql_session.refresh(action_db)

        if plant_tree_action:
            for action_specie in action_model.species:
                specieDb = SpeciesDb(
                    species_name=action_specie.name,
                    species_slug=action_specie.id,
                    sapling_number=action_specie.number,
                    action_id=action_db.action_id,
                )
                self._sql_session.add(specieDb)
                self._sql_session.commit()
                self._sql_session.refresh(specieDb)

        return action_db

    def get_all(self) -> list[ActionDb]:
        return (
            self._sql_session.query(ActionDb)
            .filter(ActionDb.deleted == 0)
            .all()
        )

    def add_action_type(self, action_id: int, action_type_model: ActionTypeAddModel) -> ActionTypeDb:
        action_type_db = ActionTypeDb(
            slug=action_type_model.slug,
            name=action_type_model.name,
            action_id=action_id,
        )
        self._sql_session.add(action_type_db)
        self._sql_session.commit()
        self._sql_session.refresh(action_type_db)

        return action_type_db

    def get_by_id(self, action_id: int) -> ActionDb:
        return (
            self._sql_session.query(ActionDb)
            .filter(ActionDb.action_id == action_id)
            .first()
        )

    def add_action_zone(self, action_id: int, zone_id: int) -> ActionZoneMTM:
        action_zone_db = ActionZoneMTM(
            action_id=action_id,
            zone_id=zone_id,
        )
        self._sql_session.add(action_zone_db)
        self._sql_session.commit()
        self._sql_session.refresh(action_zone_db)

        return action_zone_db

    def get_by_project_id(self, project_id: int) -> list[ActionDb]:
        return (
            self._sql_session.query(ActionDb)
            .filter(ActionDb.project_id == project_id)
            .filter(ActionDb.deleted == 0)
            .all()
        )

    def get_action_types_by_action_id(self, action_id: int) -> list[ActionTypeDb]:
        return (
            self._sql_session.query(ActionTypeDb)
            .filter(ActionTypeDb.action_id == action_id)
            .all()
        )

    def get_action_zones_by_action_id(self, action_id: int) -> list[ActionZoneMTM]:
        return (
            self._sql_session.query(ActionZoneMTM)
            .filter(ActionZoneMTM.action_id == action_id)
            .all()
        )

    def get_action_zones_by_zone_id(self, zone_id: int) -> list[ActionZoneMTM]:
        return (
            self._sql_session.query(ActionZoneMTM)
            .filter(ActionZoneMTM.zone_id == zone_id)
            .all()
        )
