from fastapi import APIRouter, Depends

from app.controller.schema.action_shema import ActionResponseModel
from app.controller.schema.audit_action_shema import AuditActionResponseModel
from app.controller.schema.general_shema import MessageResponseModel, Stats
from app.controller.schema.project_shema import (
    HumidityReturnModel,
    ProjectResponseModel,
)
from app.controller.schema.species_schema import SpecieInfo
from app.controller.schema.tree_hole_detection import TreeHoleDetectionReturn, TreeResponse
from app.database.sql.models import TreeDb
from app.services.routes import project_service

project_router = APIRouter(prefix="/project", tags=["project"])


@project_router.get(
    "/stats",
    summary="Projects stats",
    response_model=Stats,
    tags=["project"],
)
def get_stats(
    response: Stats = Depends(project_service.get_stats),
) -> Stats:
    return response


@project_router.post(
    "",
    status_code=201,
    summary="Create a project",
    response_model=MessageResponseModel,
    tags=["project"],
)
def create_project(
    response: MessageResponseModel = Depends(project_service.create_project),
) -> MessageResponseModel:
    return response


@project_router.get(
    "/species-info",
    summary="Get species info",
    response_model=dict[str, SpecieInfo],
    tags=["species"],
)
def species_info(
    response: dict[str, SpecieInfo] = Depends(project_service.get_species_info),
) -> dict[str, SpecieInfo]:
    return response


@project_router.put(
    "/species-info",
    summary="Update species info",
    status_code=200,
    response_model=MessageResponseModel,
    tags=["species"],
)
def update_species_info(
    response: MessageResponseModel = Depends(project_service.update_species_info),
) -> MessageResponseModel:
    return response


@project_router.put(
    "/{project_id}",
    summary="Update a project",
    status_code=200,
    response_model=MessageResponseModel,
    tags=["project"],
)
def update_project(
    response: MessageResponseModel = Depends(project_service.update_project),
) -> MessageResponseModel:
    return response


@project_router.get(
    "",
    summary="Get all projects",
    response_model=list[ProjectResponseModel],
    tags=["project"],
)
def get_all_projects(
    response: list[ProjectResponseModel] = Depends(project_service.get_all_projects),
) -> list[ProjectResponseModel]:
    return response


@project_router.get(
    "/{project_id}",
    summary="Get project by id",
    response_model=ProjectResponseModel,
    tags=["project"],
)
def get_project(
    response: ProjectResponseModel = Depends(project_service.get_project),
) -> ProjectResponseModel:
    return response


@project_router.delete(
    "/{project_id}",
    summary="Delete a project",
    response_model=MessageResponseModel,
    tags=["project"],
)
def delete_project(
    response: MessageResponseModel = Depends(project_service.delete_project),
) -> MessageResponseModel:
    return response


@project_router.post(
    "/{project_id}/zone",
    status_code=201,
    summary="Create a zone for a project",
    response_model=MessageResponseModel,
    tags=["project"],
)
def create_zone(
    response: MessageResponseModel = Depends(project_service.create_zone),
) -> MessageResponseModel:
    return response


@project_router.delete(
    "/zone/{zone_id}",
    summary="Delete a zone",
    response_model=MessageResponseModel,
    tags=["zone"],
)
def delete_zone(
    response: MessageResponseModel = Depends(project_service.delete_zone),
) -> MessageResponseModel:
    return response


@project_router.post(
    "/{project_id}/action",
    status_code=201,
    summary="Add an action to a project",
    response_model=MessageResponseModel,
    tags=["project"],
)
def add_action(
    response: MessageResponseModel = Depends(project_service.add_action),
) -> MessageResponseModel:
    return response


@project_router.get(
    "/{project_id}/action",
    summary="Get all actions of a project",
    response_model=list[ActionResponseModel],
    tags=["project"],
)
def get_actions(
    response: list[ActionResponseModel] = Depends(project_service.get_actions),
) -> list[ActionResponseModel]:
    return response


@project_router.put(
    "/zone/{zone_id}",
    summary="Update a zone",
    status_code=200,
    response_model=MessageResponseModel,
    tags=["zone"],
)
def update_zone(
    response: MessageResponseModel = Depends(project_service.update_zone),
) -> MessageResponseModel:
    return response


@project_router.post(
    "/{project_id}/audit-action",
    summary="Add an audit action to a project",
    status_code=201,
    response_model=MessageResponseModel,
    tags=["project"],
)
def add_audit_action(
    response: MessageResponseModel = Depends(project_service.add_audit_action),
) -> MessageResponseModel:
    return response


@project_router.get(
    "/{project_id}/audit-action",
    summary="Get all audit actions of a project",
    response_model=list[AuditActionResponseModel],
    tags=["project"],
)
def get_audit_actions(
    response: list[AuditActionResponseModel] = Depends(
        project_service.get_audit_actions
    ),
) -> list[AuditActionResponseModel]:
    return response


@project_router.post(
    "/process-humidity",
    summary="Get humidity and process it",
    status_code=200,
    response_model=MessageResponseModel,
    tags=["project"],
)
def process_humidity(
    response: MessageResponseModel = Depends(project_service.process_humidity),
) -> MessageResponseModel:
    return response


@project_router.get(
    "/{project_id}/humidity",
    summary="Get all humidities of a project",
    response_model=list[HumidityReturnModel],
    tags=["project"],
)
def get_humidity(
    response: list[HumidityReturnModel] = Depends(project_service.get_humidity),
) -> list[HumidityReturnModel]:
    return response


@project_router.get(
    "/{project_id}/image",
    summary="Get moisture of a project",
    response_model=None,
    tags=["project"],
)
def get_image(
    response=Depends(project_service.get_humidity_image),
):
    return response


# tree / hole detection
@project_router.post(
    "/{project_id}/tree-hole-detection",
    summary="Get tree and hole detection",
    response_model=MessageResponseModel,
    tags=["project"],
)
def tree_hole_detection(
    response: MessageResponseModel = Depends(project_service.tree_hole_detection),
) -> MessageResponseModel:
    return response


@project_router.get(
    "/{project_id}/tree-hole-detection",
    summary="Get all tree and hole detection",
    response_model=list[TreeHoleDetectionReturn],
    tags=["project"],
)
def get_tree_hole_detection(
    response: list[TreeHoleDetectionReturn] = Depends(
        project_service.get_tree_hole_detection
    ),
) -> list[TreeHoleDetectionReturn]:
    return response


@project_router.get(
    "/{project_id}/trees",
    summary="Get trees by project",
    response_model=list[TreeResponse], 
    tags=["project"],
)
def get_trees_by_project(
    response: list[TreeDb] = Depends(project_service.get_trees_by_project),
) -> list[TreeResponse]:
    return response


@project_router.get(
    "/{project_id}/trees/not-assigned/number",
    summary="Get trees not assigned by project",
    response_model=int, 
    tags=["project"],
)
def get_trees_not_assigned_by_project(
    response: int = Depends(project_service.get_number_of_trees_available),
) -> int:
    return response


@project_router.post(
    "/{project_id}/assign-trees/{partner_id}/{number_of_trees}",
    summary="Add trees to a project",
    response_model=MessageResponseModel,
    tags=["project"],
)
def assign_trees_to_partner(
    response: MessageResponseModel = Depends(project_service.assign_trees_to_partner),
) -> MessageResponseModel:
    return response
