from typing import TypeVar, Type

from fastapi import Depends
from sqlalchemy.orm import Session

from app.database.sql import sqlDBManager
from app.security.auth_handler import AuthHandler

TRepo = TypeVar("TRepo", bound="BaseRepo")


class CallSession:
    auth_handler = AuthHandler()

    def __init__(
        self,
        sql_session: Session,
        user_id: int = None,
        partner_id: str = None,
    ):
        self.sql_session = sql_session
        self.user_id = user_id
        self.partner_id = partner_id
        # a dict of repos ( key: repo_class, value: repo_instance )
        self._repos = {}

    def __getitem__(self, repo_type: Type[TRepo]) -> TRepo:
        """Get a repo instance.

        If the repo instance does not exist, create it.
        """
        if repo_type not in self._repos:
            self._repos[repo_type] = repo_type(self)
        return self._repos[repo_type]


def session_maker(auth: bool = True):
    """Return a function that will create a session on call

    Args:
        auth: If True, the function requires an auth token
    """

    # creates a session that requires an auth token
    def auth_wrapper(user_info=Depends(CallSession.auth_handler.auth_wrapper)):
        user_id, partner_id = user_info
        with sqlDBManager.session() as sql_session:
            yield CallSession(
                sql_session=sql_session,
                user_id=user_id,
                partner_id=partner_id,
            )

    # creates a session that does not require an auth token
    def simple_wrapper():
        with sqlDBManager.session() as sql_session:
            yield CallSession(sql_session=sql_session)

    # return the correct session maker
    return auth_wrapper if auth else simple_wrapper
